#----------------------------------------------------------------------
#        Statistics and Probability for Life Sciences
#                     R functions
#----------------------------------------------------------------------

#----------------------------------------------------------------------
# normal.fluctuation:  fluctuation intervals for normal distributions
# student.fluctuation: fluctuation intervals for Student distributions
# chisq.fluctuation:   fluctuation intervals for chi-squared distributions
# estim.unif:          estimates for the bound of a uniform distribution
# estim.normal:        estimates for the parameters of a normal distribution
# confint.normal:      simulated confidence intervals for a normal distribution
# normal.pvalue:       p-value for a normal variable
# sd.test:             test for the value of a standard deviation
# slreg:               simple linear regression
# slope.test:          tests on the slope of a simple linear regression
# prediction.test:     tests on a mean or predicted value in a simple linear regression
#----------------------------------------------------------------------

normal.fluctuation <- function(p,x,mean=0,sd=1,alternative="two.sided"){
# plots the density of a normal distribution;
# shades fluctuation interval with level p or bounds x.
#
rapp <- 10
if(!missing(x)){
   if(length(x)==2){
      p <- pnorm(max(x),mean,sd)-pnorm(min(x),mean,sd)
      xlim <- c(min(x),max(x))
      xout <- xlim
      if(prod(xlim)<0){
         yp <- (dnorm(mean,mean,sd)+sum(dnorm(xlim,mean,sd)))/3
         }else{yp <- mean(dnorm(xlim,mean,sd))}
      xticks <- xlim
      xtickslab <- as.character(round(xticks,4))
   }else{
      switch(alternative,
         "two.sided"={
	    xlim <- mean+c(-1,1)*abs(x-mean)
	    xout <- xlim
            p <- pnorm(xlim[2],mean,sd)-pnorm(xlim[1],mean,sd)
            yp <- (dnorm(mean,mean,sd)+sum(dnorm(xlim,mean,sd)))/3
	    xticks <- xlim
	    xtickslab <- as.character(round(xticks,4))},
         "less"={
            p <- pnorm(x,mean,sd)
	    p2 <- p/rapp
	    xlim <- c(min(mean-3*sd,qnorm(p2,mean,sd)),x)
	    xout <- c(-Inf,x)
            if(prod(xlim)<0){
	       yp <- (dnorm(mean,mean,sd)+sum(dnorm(xlim,mean,sd)))/3
               }else{yp <- mean(dnorm(xlim,mean,sd))}
	    xticks <- x
	    xtickslab <- as.character(round(xticks,4))},
         "greater"={
            p <- pnorm(x,mean,sd,lower.tail=FALSE)
	    p2 <- p/rapp
	    xlim <- c(x,max(mean+3*sd,qnorm(p2,mean,sd,lower.tail=FALSE)))
	    xout <- c(x,Inf)
            if(prod(xlim)<0){yp <- dnorm(mean,mean,sd)/2
               }else{yp <- mean(dnorm(xlim,mean,sd))}
	    xticks <- x
	    xtickslab <- as.character(round(xticks,4))})
}}else{
if(!missing(p)){                           # given level
   switch(alternative,
      "two.sided"={
         x<- c(qnorm((1-p)/2,mean,sd),qnorm(p+(1-p)/2,mean,sd))
         xlim <- x
	 xout <- x
         yp <- (dnorm(mean,mean,sd)+sum(dnorm(xlim,mean,sd)))/3
         xticks <- xlim
         xtickslab <- as.character(round(xticks,4))},
      "less"={
         x <- qnorm(p,mean,sd)
         p2 <- p/rapp
	 xlim <- c(min(mean-3*sd,qnorm(p2,mean,sd)),x)
	 xout <- c(-Inf,x)
         yp <- min(dnorm(x,mean,sd)/2)
	 xticks <- x
	 xtickslab <- as.character(round(xticks,4))},
      "greater"={
         x <- qnorm(p,mean,sd,lower.tail=FALSE)
         p2 <- p/rapp
         xlim <- c(x,max(mean+3*sd,qnorm(p2,mean,sd,lower.tail=FALSE)))
	 xout <- c(x,Inf)
         yp <- min(dnorm(x,mean,sd)/2)
         xticks <- x
	 xtickslab <- as.character(round(xticks,4))})}
else{stop("give x or p")}}
if(p>1e-3){ptxt <- as.character(round(p,4))
   }else{ptxt <- as.character(signif(p,3))}
dens <- function(y){dnorm(y,mean,sd)}
{curve(dens,                                # plot density
       xlim=xlim, col="blue",lwd=2,
       xlab="x",ylab="density",axes=FALSE,
       main="normal distribution",cex.main=1.5)}
abline(h=0)
                                            # add mean and sd as subtitle
mtext(paste("mean =",as.character(mean),"   sd =",as.character(sd)),cex=1.2)
                                            # add x-axis with ticks
axis(1,at=xticks,labels=xtickslab)
                                            # prepare shaded area
coordx <- seq(xlim[1],xlim[2],length.out=100)
coordy <- dnorm(coordx,mean,sd)             # ordinates
coordx <- c(coordx,rev(xlim))               # close polygon
coordy <- c(coordy,0,0)
polygon(coordx,coordy,col="skyblue")        # fill area
                                            # add probability as text
text(x=mean(xlim),y=yp,labels=ptxt,cex=1.5)
return(list(x=xout,p=p))
}                                           # end function normal.fluctuation

student.fluctuation <- function(p,x,df,alternative="two.sided"){
# plots the density of a Student T distribution;
# shades fluctuation interval with level p or bounds x.
#
rapp <- 10
if(!missing(x)){
   if(length(x)==2){
      p <- pt(max(x),df)-pt(min(x),df)
      xlim <- c(min(x),max(x))
      xout <- xlim
      if(prod(xlim)<0){
         yp <- (dt(0,df)+sum(dt(xlim,df)))/3
         }else{yp <- mean(dt(xlim,df))}
      xticks <- xlim
      xtickslab <- as.character(round(xticks,4))
   }else{
      switch(alternative,
         "two.sided"={
	    xlim <- c(-1,1)*abs(x)
	    xout <- xlim
            p <- pt(xlim[2],df)-pt(xlim[1],df)
            yp <- (dt(0,df)+sum(dt(xlim,df)))/3
	    xticks <- xlim
	    xtickslab <- as.character(round(xticks,4))},
         "less"={
            p <- pt(x,df)
	    p2 <- p/rapp
	    xlim <- c(min(-3,qt(p2,df)),x)
	    xout <- c(-Inf,x)
            if(prod(xlim)<0){
	       yp <- (dt(0,df)+sum(dt(xlim,df)))/3
               }else{yp <- mean(dt(xlim,df))}
	    xticks <- x
	    xtickslab <- as.character(round(xticks,4))},
         "greater"={
            p <- pt(x,df,lower.tail=FALSE)
	    p2 <- p/rapp
	    xlim <- c(x,max(3,qt(p2,df,lower.tail=FALSE)))
	    xout <- c(x,Inf)
            if(prod(xlim)<0){yp <- dt(0,df)/2
               }else{yp <- mean(dt(xlim,df))}
	    xticks <- x
	    xtickslab <- as.character(round(xticks,4))})
}}else{
if(!missing(p)){                           # given level
   switch(alternative,
      "two.sided"={
         x<- c(qt((1-p)/2,df),qt(p+(1-p)/2,df))
         xlim <- x
	 xout <- x
         yp <- (dt(0,df)+sum(dt(xlim,df)))/3
         xticks <- xlim
         xtickslab <- as.character(round(xticks,4))},
      "less"={
         x <- qt(p,df)
         p2 <- p/rapp
	 xlim <- c(min(-3,qt(p2,df)),x)
	 xout <- c(-Inf,x)
         yp <- min(dt(x,df)/2)
	 xticks <- x
	 xtickslab <- as.character(round(xticks,4))},
      "greater"={
         x <- qt(p,df,lower.tail=FALSE)
         p2 <- p/rapp
         xlim <- c(x,max(3,qt(p2,df,lower.tail=FALSE)))
	 xout <- c(x,Inf)
         yp <- min(dt(x,df)/2)
         xticks <- x
	 xtickslab <- as.character(round(xticks,4))})}
else{stop("give x or p")}}
if(p>1e-3){ptxt <- as.character(round(p,4))
   }else{ptxt <- as.character(signif(p,3))}
dens <- function(y){dt(y,df)}
{curve(dens,                                # plot density
       xlim=xlim, col="blue",lwd=2,
       xlab="x",ylab="density",axes=FALSE,
       main="Student distribution",cex.main=1.5)}
abline(h=0)
                                            # add df as subtitle
mtext(paste("df =",as.character(df)),cex=1.2)
                                            # add x-axis with ticks
axis(1,at=xticks,labels=xtickslab)
                                            # prepare shaded area
coordx <- seq(xlim[1],xlim[2],length.out=100)
coordy <- dt(coordx,df)                     # ordinates
coordx <- c(coordx,rev(xlim))               # close polygon
coordy <- c(coordy,0,0)
polygon(coordx,coordy,col="skyblue")        # fill area
                                            # add probability as text
text(x=mean(xlim),y=yp,labels=ptxt,cex=1.5)
return(list(x=xout,p=p))
}                                           # end function student.fluctuation

chisq.fluctuation <- function(p,x,df,alternative="two.sided"){
# plots the density of a chi-squared distribution;
# shades fluctuation interval with level p or bounds x.
#
rapp <- 10
if(!missing(x)){
   if(length(x)==2){
      p <- pchisq(max(x),df)-pchisq(min(x),df)
      xlim <- c(min(x),max(x))
      xout <- xlim
      if(prod(xlim-df)<0){
         yp <- (dchisq(df,df)+sum(dchisq(xlim,df)))/3
         }else{yp <- mean(dchisq(xlim,df))}
      xticks <- xlim
      xtickslab <- as.character(round(xticks,4))
   }else{
      switch(alternative,
         "two.sided"={stop("give 2 values for x")},
         "less"={
            p <- pchisq(x,df)
	    p2 <- p/rapp
	    xlim <- c(0,x)
	    xout <- c(0,x)
            if(prod(xlim-df)<0){
	       yp <- (dchisq(df,df)+sum(dchisq(xlim,df)))/3
               }else{yp <- mean(dchisq(xlim,df))}
	    xticks <- x
	    xtickslab <- as.character(round(xticks,4))},
         "greater"={
            p <- pchisq(x,df,lower.tail=FALSE)
	    p2 <- p/rapp
	    xlim <- c(x,max(3*df,qchisq(p2,df,lower.tail=FALSE)))
	    xout <- c(x,Inf)
            if(prod(xlim-df)<0){yp <- dchisq(df,df)/2
               }else{yp <- mean(dchisq(xlim,df))}
	    xticks <- x
	    xtickslab <- as.character(round(xticks,4))})
}}else{
if(!missing(p)){                           # given level
   switch(alternative,
      "two.sided"={
         x<- c(qchisq((1-p)/2,df),qchisq(p+(1-p)/2,df))
         xlim <- x
	 xout <- x
         yp <- (dchisq(df,df)+sum(dchisq(xlim,df)))/3
         xticks <- xlim
         xtickslab <- as.character(round(xticks,4))},
      "less"={
         x <- qchisq(p,df)
         p2 <- p/rapp
	 xlim <- c(0,x)
	 xout <- c(0,x)
	 if(x>df){yp <- dchisq(df,df)/2
            }else{yp <- dchisq(x,df)/2}
	 xticks <- x
	 xtickslab <- as.character(round(xticks,4))},
      "greater"={
         x <- qchisq(p,df,lower.tail=FALSE)
         p2 <- p/rapp
         xlim <- c(x,max(3*df,qchisq(p2,df,lower.tail=FALSE)))
	 xout <- c(x,Inf)
	 if(x<df){yp <- dchisq(df,df)/2
            }else{yp <- dchisq(x,df)/2}
         xticks <- x
	 xtickslab <- as.character(round(xticks,4))})}
else{stop("give x or p")}}
if(p>1e-3){ptxt <- as.character(round(p,4))
   }else{ptxt <- as.character(signif(p,3))}
dens <- function(y){dchisq(y,df)}
{curve(dens,                                # plot density
       xlim=xlim, col="blue",lwd=2,
       xlab="x",ylab="density",axes=FALSE,
       main="chi-squared distribution",cex.main=1.5)}
abline(h=0)
                                            # add df as subtitle
mtext(paste("df =",as.character(df)),cex=1.2)
                                            # add x-axis with ticks
axis(1,at=xticks,labels=xtickslab)
                                            # prepare shaded area
coordx <- seq(xlim[1],xlim[2],length.out=100)
coordy <- dchisq(coordx,df)                  # ordinates
coordx <- c(coordx,rev(xlim))               # close polygon
coordy <- c(coordy,0,0)
polygon(coordx,coordy,col="skyblue")        # fill area
                                            # add probability as text
text(x=mean(xlim),y=yp,labels=ptxt,cex=1.5)
return(list(x=xout,p=p))
}                                           # end function chisq.fluctuation

estim.unif <- function(N,n,theta){
#   Four estimators of theta for the uniform distribution over
#   the interval (0,theta).
#
X <- runif(n*N,0,theta)                     # all values
X <- matrix(X,N,n)                          # N samples as rows
T1 <- 2*rowMeans(X)                         # mean
T2 <- 2*apply(X,1,median)                   # median
T3 <- apply(X,1,max)                        # maximum
T4 <- T3*(n+1)/n                            # maximum corrected
Ta <- cbind(T1,T2,T3,T4)                    # matrix of estimates
boxplot(data.frame(Ta))                     # box plots
abline(h=theta,col="red")                   # true value
MSE <- colMeans((Ta-theta)^2)               # mean squared errors
return(MSE)
}                                           # end function estim.unif

estim.normal <- function(N,n,mu,sigma){
#   Two estimators of mu, two estimators of sigma,
#   for the normal distribution with parameters mu and sigma.
#
X <- rnorm(n*N,mu,sigma)                    # all values
X <- matrix(X,N,n)                          # N samples as rows
Tmu1 <- rowMeans(X)                         # mean
Tmu2 <- apply(X,1,median)                   # median
Tsigma1 <- apply(X,1,sd)                    # standard deviation
Tsigma2 <- apply(X,1,IQR)                   # interquartile range
Tsigma2 <- Tsigma2/(qnorm(0.75)-qnorm(0.25))# corrected
par(mfrow=c(1,2))                           # split graphic window
Tmu <- cbind(Tmu1,Tmu2)                     # matrix of estimates
colnames(Tmu) <- c("mean","median")
boxplot(data.frame(Tmu),main="mu")          # box plots
abline(h=mu,col="red")                      # true value
MSEmu <- colMeans((Tmu-mu)^2)               # mean squared errors
Tsigma <- cbind(Tsigma1,Tsigma2)            # matrix of estimates
colnames(Tsigma) <- c("sd","IQR")
boxplot(data.frame(Tsigma),main="sigma")    # box plots
abline(h=sigma,col="red")                   # true value
MSEsigma <- colMeans((Tsigma-sigma)^2)      # mean squared errors
res <-  list(mu=MSEmu,sigma=MSEsigma)
return(res)
}                                           # end function estim.theta

confint.normal <- function(N,n,mu,sigma,level){
#   Confidence intervals for normal distribution.
#
al2 <- (1-level)/2                          # lower bound
ual2 <- 1-al2                               # upper bound
X <- rnorm(N*n,mean=mu,sd=sigma)              # normal random numbers
X <- matrix(X,N,n)                          # N samples as rows
Xbar <- rowMeans(X)                         # empirical means
S2 <- rowMeans(X^2)-Xbar^2                  # empirical variances
rS2  <- sqrt(S2)                            # standard-deviations
par(mfrow=c(1,3))                           # split window
yl <- paste(format(level,2,2),"Confidence Intervals")
#------------------------------------------ # CI for mu, sigma known
za <- qnorm(ual2)                           # quantile normal
amp <- za*sigma/sqrt(n)                     # amplitudes
lb <- Xbar - amp                            # lower bounds
ub <- Xbar + amp                            # upper bounds
sm1<-(length(which((lb<mu)&(ub>mu))))/N     # proportion of success
{matplot(rbind(lb,ub),rbind(1:N,1:N),       # horizontal segments
type="l",lty=1,col="blue",xlab="mu",ylab=yl,
main="mu, sigma known")}
abline(v=mu,col="red")                      # true value
#------------------------------------------ # CI or mu, sigma unknown
za <- qt(ual2,df=n-1)                       # quantile Student
amp <- za*rS2/sqrt(n-1)                     # amplitudes
lb <- Xbar - amp                            # lower bounds
ub <- Xbar + amp                            # upper bounds
sm2<-(length(which((lb<mu)&(ub>mu))))/N     # proportion of success
{matplot(rbind(lb,ub),rbind(1:N,1:N),       # horizontal segments
type="l",lty=1,col="blue",xlab="mu",ylab=yl,
main="mu, sigma unknown")}
abline(v=mu,col="red")                      # true value
#------------------------------------------ # CI for sigma
ua <- qchisq(al2,df=n-1)                    # quantile chi-squared
va <- qchisq(ual2,df=n-1)                   # quantile chi-squared
lb <- rS2*sqrt(n/va)                        # lower bounds
ub <- rS2*sqrt(n/ua)                        # upper bounds
ss<-(length(which((lb<sigma)&(ub>sigma))))/N# proportion of success
{matplot(rbind(lb,ub),rbind(1:N,1:N),       # horizontal segments
type="l",lty=1,col="blue",xlab="sigma",ylab=yl,
main="sigma")}
abline(v=sigma,col="red")                   # true value

return(c(sm1,sm2,ss))
}                                           # end function confint.normal

normal.pvalue <- function(x,mean,sd,alternative="two.sided"){
#   P-value for a normal variable.
#
switch(alternative,
   "two.sided"={
       pv <- pnorm(x,mean,sd)
       if (pv>0.5){pv <- pnorm(x,mean,sd,lower.tail=FALSE)}
       pv <- 2*pv
   },
   "less"={
       pv <-  pnorm(x,mean,sd)
   },
   "greater"={
       pv <- pnorm(x,mean,sd,lower.tail=FALSE)
   })                                       # end switch
names(pv) <- "p-value"
return(pv)
}                                           # end function normal.pvalue

sd.test <- function(X,sigma,alternative="two.sided"){
#   test for the value of a standard deviation
#
n <- length(X)                              # sample size
est <- sd(X); names(est) <- "sd"
S2 <- var(X)                                # empirical variance
Test <- (n-1)*S2/sigma^2                    # test statistic
switch(alternative,
   "two.sided"={
      if(S2<sigma^2){
      pv<-pchisq(Test,df=n-1)               # p-value
      }else{                                # p-value
      pv<-pchisq(Test,df=n-1,lower.tail=FALSE) 
      }
      pv <- 2*pv
      alt = paste("true standard deviation is not equal to",as.character(sigma))
   },
   "less"={
      pv<-pchisq(Test,df=n-1)               # p-value
      alt = paste("true standard deviation is less than",as.character(sigma))
   },
   "greater"={
      pv<-pchisq(Test,df=n-1,lower.tail=FALSE) # p-value
      alt = paste("true standard deviation is greater than",as.character(sigma))
   })                                       # end switch
   {res <- list("p-value"=pv,"X-squared"=Test,"df"=(n-1),
                "alternative"=alt,
		"estimate"=est)}
return(res)
}                                           # end function sd.test

slreg <- function(X,Y,level=0.95,xstar){
# Simple linear regression of Y onto X. Confidence and prediction
# regions; confidence and prediction intervals at xstar if given.
#
                     #----------------------- compute coefficients
n <- length(X)                              # sample size
xbar <- mean(X)                             # mean predictor
ybar <- mean(Y)                             # mean response
sx2 <- mean(X^2)-xbar^2                     # variance predictor
sy2 <- mean(Y^2)-ybar^2                     # variance response
cxy <- mean(X*Y)-xbar*ybar                  # covariance
rxy <- cxy/sqrt(sx2*sy2)                    # correlation
ahat <- cxy/sx2                             # slope
bhat <- ybar-ahat*xbar                      # intercept
sighat2 <- (n/(n-2))*sy2*(1-rxy^2)          # variance residuals
                     #----------------------- scaling for graphics
m<-min(X);M<-max(X)                         # min and max of predictor
xlim <- c(m-0.1*(M-m),M+0.1*(M-m))          # bounds for x-axis
if(!missing(xstar)){                        # if prediction requested
   if(xstar<xlim[1]){xlim[1]<-xstar}        # push bounds accordingly
   if(xstar>xlim[2]){xlim[2]<-xstar}
}                                           # end if
                     #----------------------- confidence region
p<-(1+level)/2 ;talpha<-qt(p,df=n-2)        # quantile student distribution
ubc <- function(x){                         # upper bound of confidence region
   return(ahat*x+bhat+talpha*sqrt((sighat2/(n*sx2))*(sx2+(x-xbar)^2)))}
lbc <- function(x){                         # lower bound of confidence region
   return(ahat*x+bhat-talpha*sqrt((sighat2/(n*sx2))*(sx2+(x-xbar)^2)))}
                     #----------------------- prediction region
ubp <- function(x){                         # upper bound of prediction region
   return(ahat*x+bhat+talpha*sqrt((sighat2/(n*sx2))*(sx2*(n+1)+(x-xbar)^2)))}
lbp <- function(x){                         # lower bound of prediction region
   return(ahat*x+bhat-talpha*sqrt((sighat2/(n*sx2))*(sx2*(n+1)+(x-xbar)^2)))}
                     #----------------------- bounds for graphics
m<-min(Y);M<-max(Y)                         # min and max of response
ylim <- c(m-0.1*(M-m),M+0.1*(M-m))          # bounds for y-axis
if(!missing(xstar)){                        # if prediction requested
   lystar <- lbp(xstar)                     # lower bound prediction
   uystar <- ubp(xstar)                     # upper bound prediction
   if(lystar<ylim[1]){ylim[1]<-lystar}      # push bounds accordingly
   if(uystar>ylim[2]){ylim[2]<-uystar}
}                                           # end if
                     #----------------------- plot points and regression line
{plot(X,Y,pch=19,col="black",               # plot points as black dots
   xlim=xlim,ylim=ylim)}                    # scale axes
abline(a=bhat,b=ahat,col="red",lw=3)        # plot regression line
points(xbar,ybar,pch="+",cex=4,col="red")   # mark center of gravity
if(!missing(xstar)){                        # if prediction requested
   abline(v=xstar,col="red",lty=2)          # vertical at xstar
   abline(h=ahat*xstar+bhat,col="red",lty=2)# horizontal at ystar
}                                           # end if
                     #----------------------- confidence region
curve(ubc(x),add=TRUE,col="blue",lw=2)      # plot upper bound
curve(lbc(x),add=TRUE,col="blue",lw=2)      # plot lower bound
                     #----------------------- prediction region
curve(ubp(x),add=TRUE,col="blue",lw=2,lty=2)# plot upper bound
curve(lbp(x),add=TRUE,col="blue",lw=2,lty=2)# plot lower bound
                     #----------------------- confidence interval for slope
lbs <- ahat-talpha*sqrt((sighat2/(n*sx2)))
ubs <- ahat+talpha*sqrt((sighat2/(n*sx2)))
cis <- c(lbs,ubs)
                     #----------------------- confidence interval for intercept
cii <- c(lbc(0),ubc(0))
                     #----------------------- result for slope
ress <- c(ahat,cis)
names(ress) <- c("estimate","lower","upper")
                     #----------------------- result for intercept
resi <- c(bhat,cii)
names(resi) <- c("estimate","lower","upper")
                     #----------------------- result for residuals
resr <- cor(X,Y)^2
resr <- c(resr,sighat2)
names(resr) <-  c("R-squared","variance")
                     #----------------------- format results
res <- list(intercept=resi,slope=ress,residuals=resr)
if(!missing(xstar)){                        # if prediction requested
   resp <- c(xstar,ahat*xstar+bhat,lbc(xstar),ubc(xstar),lbp(xstar),ubp(xstar))
   names(resp) <- {c("xstar","ystar","conf.lower",
                                     "conf.upper",
				     "pred.lower",
				     "pred.upper")}
   res[["prediction"]] <- resp
}                                           # end if
return(res)
}                                           # end function slreg

slope.test <- function(X,Y,slope=0,alternative="two.sided"){
# Simple linear regression of Y on X. Tests value a for slope.
# Returns the p-value, according to alternative.
#
                     #----------------------- compute coefficients
n <- length(X)                              # sample size
xbar <- mean(X)                             # mean predictor
ybar <- mean(Y)                             # mean response
sx2 <- mean(X^2)-xbar^2                     # variance predictor
sy2 <- mean(Y^2)-ybar^2                     # variance response
cxy <- mean(X*Y)-xbar*ybar                  # covariance
rxy <- cxy/sqrt(sx2*sy2)                    # correlation
ahat <- cxy/sx2                             # slope
sighat2 <- (n/(n-2))*sy2*(1-rxy^2)          # variance residuals
                     #----------------------- test statistic
Ts <- sqrt(n*sx2/sighat2)*(ahat-slope)
                     #----------------------- p-value
switch(alternative,
   "two.sided"={
       pv <- pt(Ts,df=n-2)                  # cdf Student T distribution
       if (pv>0.5){pv <- pt(Ts,df=n-2,lower.tail=FALSE)}
       pv <- 2*pv
   },
   "less"={
       pv <-  pt(Ts,df=n-2)
   },
   "greater"={
       pv <- pt(Ts,df=n-2,lower.tail=FALSE)
   })                                       # end switch
names(pv) <- "p-value"
return(pv)
}                                           # end function slope.test

prediction.test <- function(X,Y,xstar,ystar,pred=TRUE,alternative="two.sided"){
# Simple linear regression of Y on X. Tests a value ystar predicted at xstar.
#
                     #----------------------- compute coefficients
n <- length(X)                              # sample size
xbar <- mean(X)                             # mean predictor
ybar <- mean(Y)                             # mean response
sx2 <- mean(X^2)-xbar^2                     # variance predictor
sy2 <- mean(Y^2)-ybar^2                     # variance response
cxy <- mean(X*Y)-xbar*ybar                  # covariance
rxy <- cxy/sqrt(sx2*sy2)                    # correlation
ahat <- cxy/sx2                             # slope
bhat <- ybar-ahat*xbar                      # intercept
sighat2 <- (n/(n-2))*sy2*(1-rxy^2)          # variance residuals
Ystar <-  ahat*xstar+bhat                   # mean prediction
                     #----------------------- test statistic
Ts <- sqrt(n*sx2/sighat2)*((ahat*xstar+bhat)-ystar)
if(pred){
   Ts <- Ts/sqrt((n+1)*sx2+(xstar-xbar)^2)
   }else{
   Ts <- Ts/sqrt(sx2+(xstar-xbar)^2)
}                                           # end if
                     #----------------------- p-value
switch(alternative,
   "two.sided"={
       pv <- pt(Ts,df=n-2)                  # cdf Student T distribution
       if (pv>0.5){pv <- pt(Ts,df=n-2,lower.tail=FALSE)}
       pv <- 2*pv
   },
   "less"={
       pv <-  pt(Ts,df=n-2)
   },
   "greater"={
       pv <- pt(Ts,df=n-2,lower.tail=FALSE)
   })                                       # end switch
names(pv) <- "p-value"
return(pv)
}                                           # end function prediction.test

